/* batch_ScoreAnimations1-2PT-64.jsx — 64x32 2PT Animation
 * - Updates TwerkingPreComp layers (Primary/Secondary/Static) with special color logic
 * - Tints 2ptText-PreComp with Secondary * 0.8 brightness
 * - Sets TeamAbbrev text and fill to Primary color
 * - Replaces TeamLogo from Logos-Large directory
 */

(function () {
  // --- load common helpers ---
  (function(){
    var COMMON = $.getenv("AE_COMMON_JSX") || "";
    if (COMMON) {
      try { delete $.global.GL; } catch(e) { $.global.GL = undefined; }
      $.evalFile(File(COMMON));
    }
    if (!$.global.GL) throw new Error("Common library not loaded. Set AE_COMMON_JSX to gl_common.jsxinc");
  })();

  var GL = $.global.GL;

  // --- env & defaults ---
  var PROJECT   = GL.env("AE_PROJECT", null);
  var CSV_PATH  = GL.env("AE_CSV", null);
  var COMP_NAME = GL.env("AE_COMP", "64x32 - 2PT");

  // Layer names
  var TWERKING_PRECOMP   = GL.cleanValue(GL.env("AE_TWERKING_PRECOMP", "TwerkingPreComp"));
  var TWO_PT_TEXT_PRECOMP = GL.cleanValue(GL.env("AE_2PT_TEXT_PRECOMP", "2ptText-PreComp"));
  var TEAMABBREV_LAYER   = GL.cleanValue(GL.env("AE_TEAMABBREV_LAYER", "TeamAbbrev"));
  var LOGO_LAYER         = GL.cleanValue(GL.env("AE_LOGO_LAYER", "TeamLogo"));

  var LEAGUE    = GL.env("AE_LEAGUE", "");
  var LIMIT_STR = GL.env("AE_LIMIT", "");
  var LIMIT     = (LIMIT_STR && !isNaN(parseInt(LIMIT_STR,10))) ? parseInt(LIMIT_STR,10) : null;

  var OUTDIR    = GL.env("AE_OUTDIR", "");
  var PATH_TPL  = GL.env("AE_PATH_TEMPLATE", "{league}");
  var ANIM_NAME = GL.env("AE_ANIM", "_2PT_1_1");
  var RS_TPL    = GL.env("AE_RS_TEMPLATE", "Best Settings");
  var OM_TPL    = GL.env("AE_OM_TEMPLATE", "PNG Sequence");
  var PURGE     = (GL.env("AE_PURGE_BEFORE_RENDER", "1") === "1");
  var NO_RENDER = (GL.env("AE_NO_RENDER", "0") === "1");
  var QUIT_APP  = (GL.env("AE_QUIT", "1") === "1");

  var logoOpts = {
    dir: GL.env("AE_LOGO_DIR", ""),
    tpl: GL.env("AE_LOGO_PATH_TEMPLATE", "{league}/{abbr}"),
    exts: GL.env("AE_LOGO_EXTS", "png,jpg,jpeg,svg,ai,psd")
  };

  // --- start ---
  if (app.beginSuppressDialogs) { try { app.beginSuppressDialogs(); } catch(e) {} }
  app.beginUndoGroup("2PT 64x32 - Batch Render");

  if (!PROJECT) GL.fail("AE_PROJECT env not set.");
  var aep = new File(PROJECT);
  if (!aep.exists) GL.fail("AE_PROJECT not found: " + PROJECT);
  try {
    var already = app.project && app.project.file && (app.project.file.fsName === aep.fsName);
    if (!already) app.open(aep);
  } catch(e) { app.open(aep); }

  if (!CSV_PATH) GL.fail("AE_CSV env not set.");
  if (!LEAGUE || GL.cleanValue(LEAGUE) === "") GL.fail("AE_LEAGUE is required.");

  var rows  = GL.parseCSV(GL.openRead(CSV_PATH));
  var teams = GL.buildTeams(rows, LEAGUE);
  var todo  = GL.pickTeamsLeagueOnly(teams, LEAGUE);
  if (LIMIT && todo.length > LIMIT) todo = todo.slice(0, LIMIT);
  if (!todo.length) GL.fail("No teams matched league: " + LEAGUE);

  var comp = GL.findComp(COMP_NAME);
  if (!comp) GL.fail("Comp not found: " + COMP_NAME);

  var rootOut = OUTDIR ? new Folder(OUTDIR) : (app.project.file ? app.project.file.parent : Folder.desktop);
  GL.ensureFolder(rootOut);

  GL.rqClear();

  // --- helper functions ---

  // Check if color is light (luminance > 0.5)
  function isLightColor(c) {
    var luma = 0.2126 * c[0] + 0.7152 * c[1] + 0.0722 * c[2];
    return luma > 0.5;
  }

  // Check if two colors are the same
  function colorsAreSame(c1, c2) {
    return Math.abs(c1[0] - c2[0]) < 0.02 &&
           Math.abs(c1[1] - c2[1]) < 0.02 &&
           Math.abs(c1[2] - c2[2]) < 0.02;
  }

  // Compute special colors for TwerkingPreComp
  function computeTwerkingColors(primaryRaw, secondaryRaw) {
    var p = GL.safeColor(GL.ensureNonBlack(primaryRaw, [0.7, 0.7, 0.7]));
    var s = GL.safeColor(GL.ensureNonBlack(secondaryRaw, [0.7, 0.7, 0.7]));

    // Check if colors are the same
    if (colorsAreSame(p, s)) {
      var darkGrey = [0.2, 0.2, 0.2];

      if (isLightColor(p)) {
        // Light color: Primary stays, Secondary = dark grey
        $.writeln("  → Colors are same (light): Primary stays, Secondary = dark grey");
        return { primary: p, secondary: darkGrey };
      } else {
        // Dark color: Primary = white, Secondary = dark grey
        $.writeln("  → Colors are same (dark): Primary = white, Secondary = dark grey");
        return { primary: [1, 1, 1], secondary: darkGrey };
      }
    } else {
      // Different colors: use smart colors
      $.writeln("  → Colors are different: using smart colors");
      return GL.computeSmartColors(p, s);
    }
  }

  // Scale brightness of color
  function scaleBrightness(c, factor) {
    return [
      Math.max(0, Math.min(1, c[0] * factor)),
      Math.max(0, Math.min(1, c[1] * factor)),
      Math.max(0, Math.min(1, c[2] * factor))
    ];
  }

  // Update TwerkingPreComp layers with Primary/Secondary colors
  function updateTwerkingPreComp(compItem, primaryColor, secondaryColor) {
    var layer = GL.getLayer(compItem, TWERKING_PRECOMP);
    if (!layer) {
      $.writeln("WARNING: TwerkingPreComp layer not found: " + TWERKING_PRECOMP);
      return 0;
    }

    if (!layer.source || !(layer.source instanceof CompItem)) {
      $.writeln("WARNING: TwerkingPreComp is not a precomp");
      return 0;
    }

    var precomp = layer.source;
    var updated = 0;

    // Loop through layers in precomp
    for (var i = 1; i <= precomp.numLayers; i++) {
      var subLayer = precomp.layer(i);
      var name = String(subLayer.name).toLowerCase();

      try {
        if (name.indexOf("primary") !== -1) {
          // Apply primary color
          if (GL.setTintBothColors(subLayer, primaryColor, true)) {
            $.writeln("  ✓ Tinted '" + subLayer.name + "' with Primary color");
            updated++;
          }
        } else if (name.indexOf("secondary") !== -1) {
          // Apply secondary color
          if (GL.setTintBothColors(subLayer, secondaryColor, true)) {
            $.writeln("  ✓ Tinted '" + subLayer.name + "' with Secondary color");
            updated++;
          }
        } else if (name.indexOf("static") !== -1) {
          // Leave static layer alone
          $.writeln("  → Skipped static layer: " + subLayer.name);
        }
      } catch(e) {
        $.writeln("  ✗ Error updating '" + subLayer.name + "': " + e);
      }
    }

    return updated;
  }

  // Tint 2ptText-PreComp with Secondary * 0.8
  function tint2ptTextPreComp(compItem, secondaryColor) {
    var layer = GL.getLayer(compItem, TWO_PT_TEXT_PRECOMP);
    if (!layer) {
      $.writeln("WARNING: 2ptText-PreComp layer not found: " + TWO_PT_TEXT_PRECOMP);
      return 0;
    }

    // Scale brightness to 0.8
    var dimmedColor = scaleBrightness(secondaryColor, 0.8);

    // Apply tint to the layer (not deep into precomp)
    return GL.setTintBothColors(layer, dimmedColor, /*forceAmount100*/ true);
  }

  // Set TeamAbbrev text content and fill color to primary
  function setTeamAbbrevTextAndFill(compItem, abbr, color01) {
    var layer = GL.getLayerAlt(compItem, TEAMABBREV_LAYER, ["TeamAbbrec"]);
    if (!layer) {
      $.writeln("WARNING: TeamAbbrev layer not found: " + TEAMABBREV_LAYER);
      return false;
    }

    // Set text content
    GL.setTextContent(layer, abbr);

    // Set fill color (handles text layers, shape layers, and effects)
    return GL.setTextOrFillColor(layer, color01);
  }

  // --- per team loop ---
  $.writeln("\n========================================");
  $.writeln("Processing " + todo.length + " teams for league: " + LEAGUE);
  $.writeln("Comp: " + COMP_NAME);
  $.writeln("Animation: " + ANIM_NAME);
  $.writeln("========================================\n");

  for (var i = 0; i < todo.length; i++) {
    var t = todo[i];
    $.writeln("["+(i+1)+"/"+todo.length+"] Processing: " + t.abbr + " (" + t.name + ")");

    // Compute colors with special logic for TwerkingPreComp
    var twerkColors = computeTwerkingColors(t.primary, t.secondary);
    var twerkPrimary = twerkColors.primary;
    var twerkSecondary = twerkColors.secondary;

    // Regular smart colors for TeamAbbrev
    var smart = GL.computeSmartColors(GL.safeColor(t.primary), GL.safeColor(t.secondary));
    var regularPrimary = smart.primary;
    var regularSecondary = smart.secondary;

    // 1) Update TwerkingPreComp with special colors
    var twerkUpdates = updateTwerkingPreComp(comp, twerkPrimary, twerkSecondary);
    if (twerkUpdates > 0) {
      $.writeln("  ✓ Updated TwerkingPreComp (" + twerkUpdates + " layers)");
    }

    // 2) Tint 2ptText-PreComp with Secondary * 0.8
    if (tint2ptTextPreComp(comp, regularSecondary)) {
      $.writeln("  ✓ Tinted 2ptText-PreComp (Secondary * 0.8)");
    }

    // 3) Set TeamAbbrev text and fill to Primary
    if (setTeamAbbrevTextAndFill(comp, t.abbr, regularPrimary)) {
      $.writeln("  ✓ Set TeamAbbrev: " + t.abbr + " (Primary color)");
    }

    // 4) Replace TeamLogo from Logos-Large
    var logoFile = GL.findLogoFile(t.league, t.abbr, logoOpts);
    if (logoFile) {
      $.writeln("  → Found logo: " + logoFile.fsName);

      // Import footage once
      var footage = null;
      try {
        var io = new ImportOptions(logoFile);
        if (io.canImportAs(ImportAsType.FOOTAGE)) {
          footage = app.project.importFile(io);
          $.writeln("  ✓ Imported logo: " + footage.name);
        }
      } catch(e) {
        $.writeln("  ✗ ERROR importing logo: " + e.toString());
      }

      if (footage) {
        // Check layer exists
        var logoLayer = GL.getLayer(comp, LOGO_LAYER);
        if (!logoLayer) {
          $.writeln("  ✗ Layer '" + LOGO_LAYER + "' NOT FOUND in comp");
        } else {
          // Check if it's a precomp
          if (logoLayer.source && (logoLayer.source instanceof CompItem)) {
            $.writeln("  → '" + LOGO_LAYER + "' is a precomp, looking inside...");
            var precomp = logoLayer.source;
            var replaced = 0;

            // Replace all footage layers inside the precomp
            for (var L = 1; L <= precomp.numLayers; L++) {
              var subLayer = precomp.layer(L);
              try {
                if (subLayer.source && (subLayer.source instanceof FootageItem)) {
                  subLayer.replaceSource(footage, false);
                  $.writeln("  ✓ Replaced '" + subLayer.name + "' inside precomp");
                  replaced++;
                }
              } catch(e) {
                $.writeln("  ✗ Failed to replace '" + subLayer.name + "': " + e);
              }
            }

            if (replaced === 0) {
              $.writeln("  ✗ No footage layers found inside '" + LOGO_LAYER + "' precomp");
            }
          } else if (logoLayer.source) {
            // Direct footage layer
            try {
              logoLayer.replaceSource(footage, false);
              $.writeln("  ✓ Replaced TeamLogo for " + t.abbr);
            } catch(e) {
              $.writeln("  ✗ ERROR replacing logo: " + e.toString());
            }
          } else {
            $.writeln("  ✗ Layer '" + LOGO_LAYER + "' has no source (cannot replace)");
          }
        }
      }
    } else {
      $.writeln("  ✗ Logo file not found for " + t.abbr + " at: " + logoOpts.dir + "/" + logoOpts.tpl.replace("{league}", t.league).replace("{abbr}", t.abbr));
    }

    // Pixel snap for crisp rendering
    GL.snapCompTextForPixelArt(comp);

    if (PURGE && app.purge) {
      try { app.purge(PurgeTarget.ALL_CACHES); } catch(e) {}
    }

    // Render
    if (!NO_RENDER) {
      var lc = GL.leagueAndConfForPath(t.league, t.conference);
      var paths = GL.outPaths(rootOut, PATH_TPL, lc.base, t.abbr, ANIM_NAME, lc.conf, t.espn_team_id);
      $.writeln("  → Output: " + paths.file.fsName);
      GL.rqRenderTo(comp, RS_TPL, OM_TPL, paths.file);
    }

    // Clean up logo from project after render
    // DISABLED FOR TESTING - uncomment when working
    // var removed = GL.removeLogoByAbbr(t.abbr);
    // if (removed > 0) {
    //   $.writeln("  ✓ Removed " + removed + " logo(s) for " + t.abbr);
    // }

    $.writeln("");
  }

  $.writeln("========================================");
  $.writeln("Batch complete: " + todo.length + " teams processed");
  $.writeln("========================================");

  app.endUndoGroup();
  if (app.endSuppressDialogs) { try { app.endSuppressDialogs(); } catch(e) {} }
  if (QUIT_APP) app.quit();
})();
